<?php

require_once(__DIR__ . '/util/utils.php');

function CheckContentsGetTime($name){
    try{
        $ret = TRUE;
        $filename =  __DIR__.'/../contents/'.$name.'.txt';
        $fp = fopen($filename, 'a+');
        $contents = fread($fp, filesize($filename));

        $ret = preg_match('/^\d*/',$contents, $matches);

        $tm = $matches[0];
        if($tm != ""){
            $p = new Datetime();
            $t = clone $p;
            $p->setTimestamp($tm);
            $ret = checkdate($p->format("m"), $p->format("d"), $p->format("Y"));
            if(!$ret) throw new Exception('invalid timestamp');
            if($p->getTimestamp() > $t->getTimestamp()){
                $ret = FALSE;
            }
        }

        return $ret;
    }catch(Exception $e){
        throw new Exception($e->getMessage());
    }
}



function WriteNewsContentsForBuffer($content,$min){
    try{
        $filename =  __DIR__.'/../contents/news.txt';
        chmod($filename, 0777);
        chown($filename, 'apache');
        $fp = fopen($filename, 'w+');
        $contents = fread($fp, filesize($filename));
        $t = new Datetime();
        if(!$min){
            throw new Exception("Failed to get time.");
        }
        $tm = $t->modify("+ ".$min." minute")->getTimestamp();

        if ($fp){
            if (!flock($fp, LOCK_EX)) throw new Exception("Failed to lock NEWS buffer file.");

            if (fwrite($fp,  $tm.'_'.$content) === FALSE) throw new Exception("Failed to write NEWS buffer file.");

            flock($fp, LOCK_UN);
        }

        $flag = fclose($fp);
        if(!$flag) throw new Exception("Failed to close.");
    }catch(Exception $e){
        //         var_dump($e->getMessage());exit;
        throw new Exception($e->getMessage());
    }
}


function GetNewsContents(){

    try{

        $url = 'https://newsapi.org/v2/top-headlines?country=jp&category=business&apiKey='.API_NEWS_KEY;
        $header = [
            "User-Agent:Linux"
        ];
        $ch = curl_init();
        curl_setopt($ch,CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        curl_setopt($ch,CURLOPT_SSL_VERIFYPEER, FALSE);  // 証明書の検証を無効化
        curl_setopt($ch,CURLOPT_SSL_VERIFYHOST, FALSE);  // 証明書の検証を無効化
        curl_setopt($ch,CURLOPT_RETURNTRANSFER, TRUE); // 返り値を文字列に変更
        curl_setopt($ch,CURLOPT_FOLLOWLOCATION, TRUE); // Locationヘッダを追跡

        $jsonData= curl_exec($ch);

        curl_close($ch);

        //         $jsonData = '{"status":"ok","totalResults":70,"articles":[{"source":{"id":null,"name":"Nhk.or.jp"},"author":null,"title":"c1西武ハイヤー 雇用調整助成金の一部 会社の特別利益として計上 - NHK NEWS WEB","description":"私鉄大手、西武ホールディングスの傘下の「西武ハイヤー」が、国から受け取った雇用調整助成金の一部を休業手当にあてず、差額の…","url":"https://www3.nhk.or.jp/news/html/20210419/k10012983151000.html","urlToImage":"https://www3.nhk.or.jp/news/html/20210419/K10012983151_2104191258_2104191302_01_02.jpg","publishedAt":"2021-04-19T06:24:06Z","content":""},{"source":{"id":null,"name":"Impress.co.jp"},"author":null,"title":"楽天モバイルから「Rakuten BIG s」、ミリ波対応で4眼カメラ搭載 - ケータイ Watch","description":"楽天モバイルは、同社オリジナルのAndroidスマートフォン「Rakuten BIG s」を4月21日から順次発売する。価格は5万980円（税込）。","url":"https://k-tai.watch.impress.co.jp/docs/news/1319551.html","urlToImage":"https://k-tai.watch.impress.co.jp/img/ktw/list/1319/551/sub.png","publishedAt":"2021-04-19T05:40:20Z","content":"AndroidRakuten BIG s4215980 \r\n Rakuten BIG s6.4EL48MP45GeSIM \r\n 28MP8MP48MP2MP2020Rakuten BIG \r\n Rakuten Browser \r\n 34215 \r\n 56"},{"source":{"id":null,"name":"Itmedia.co.jp"},"author":"ITmedia","title":"COCOA元請けのパーソルP&Tが1200万円自主返納 対応に疑問の声も - ITmedia","description":"接触確認アプリ「COCOA」の不具合を巡り、アプリ開発の元請けであるパーソルプロセス&テクノロジー（パーソルP&T、東京都江東区）が受託金額の一部を自主返納すると発表。これに対し、Twitterでは疑問の声が上がっている。","url":"https://www.itmedia.co.jp/news/articles/2104/19/news099.html","urlToImage":"https://image.itmedia.co.jp/news/articles/2104/19/cover_news099.jpg","publishedAt":"2021-04-19T05:32:00Z","content":"COCOA&amp;P&amp;T4161200\r\n2020539000COCOAFIXER\r\n4Android\r\nCopyright © ITmedia, Inc. All Rights Reserved."},{"source":{"id":null,"name":"Nikkei.com"},"author":"日本経済新聞社","title":"輸出の中国比率、初の2割超 米中対立がリスクに - 日本経済新聞","description":"財務省が19日発表した2020年度の貿易統計速報は、輸出入で中国への依存を強める日本経済の現状を映した。輸出に占める中国向けの比率は22.9%と初めて2割を超えて米国を抜き、10年ぶりに過去最高を更新した。深刻化する米中対立が貿易取引に波及すれば、新型コロナウイルス禍からの景気回復を阻むリスクになりかねない。20年度の輸出は全体で8.4%減の69兆4873億円。米リーマン危機後だった09年度の","url":"https://www.nikkei.com/article/DGXZQOUA181JM0Y1A410C2000000/","urlToImage":"https://article-image-ix.nikkei.com/https%3A%2F%2Fimgix-proxy.n8s.jp%2FDSXZQO0520129019042021000000-1.jpg?auto=format%2Ccompress&ch=Width%2CDPR&q=45&fit=crop&bg=FFFFFF&w=1200&h=630&fp_x=0.5&fp_y=0.5&fp_z=1&crop=focalpoint&ixlib=js-1.4.1&s=a956eee6cf173d169c446c3b8cb57620","publishedAt":"2021-04-19T05:00:00Z","content":null},{"source":{"id":null,"name":"Bloomberg.co.jp"},"author":null,"title":"米国債、弱気派には厳しい環境－10年物利回り1.2％に低下の観測も - ブルームバーグ","description":"米国債の下落を見込むポジションを構築したり現金に退避して値下がりを待っている投資家にとって先週は、経済指標が米景気回復の勢いを示したにもかかわらず、厳しい１週間だった。","url":"https://www.bloomberg.co.jp/news/articles/2021-04-19/QRSE91T1UM1E01","urlToImage":"https://assets.bwbx.io/images/users/iqjWHBFdfxIU/imD2Wxz_vY2Q/v2/-1x-1.png","publishedAt":"2021-04-19T04:43:19Z","content":"米国債の下落を見込むポジションを構築したり現金に退避して値下がりを待っている投資家にとって先週は、経済指標が米景気回復の\r\n勢いを示したにもかかわらず、厳しい１週間だった。\r\n　　米国債の先行きを巡る議論は、終わったというには\r\n程遠い。2021年に入って主流だった弱気な見方は先週、週ベースで昨年８月以来の大幅値上がりから打撃を受けた。利回りが一時的にさらに低下する可能性を指摘するストラテジスト… [+1186 chars]"},{"source":{"id":null,"name":"Rocketnews24.com"},"author":"https://www.facebook.com/neko.ahiru.1","title":"【速報】松屋の「ごろチキ」が復活！ 繰り返す、松屋の「ごろごろ煮込みチキンカレー」が復活だァァァァアアア!! 4月27日から！ - ロケットニュース24","description":"速報ォォォォオオオオ！　とんでもねぇ速報だァァァァァアアアアア!!    松屋の「ごろチキ」が復活ッ！　松屋の「ごろチキ」が復活ッ!!　松屋の「ごろチキ」が復活ッ！　 …","url":"https://rocketnews24.com/2021/04/19/1481889/","urlToImage":"https://rocketnews24.com/wp-content/uploads/sites/2/2019/10/014.jpg?w=1200&h=630&crop=1","publishedAt":"2021-04-19T04:34:32Z","content":null},{"source":{"id":null,"name":"Itmedia.co.jp"},"author":"ITmedia","title":"血中酸素酸素レベル測れるスマートウォッチ「OPPO Band Style」4480円で登場 - ITmedia","description":"血中酸素酸素レベル測定機能などを備えた多機能スマートウォッチ「OPPO Band Style」が登場。4480円（税込）と低価格だ。","url":"https://www.itmedia.co.jp/news/articles/2104/19/news088.html","urlToImage":"https://image.itmedia.co.jp/news/articles/2104/19/cover_news088.jpg","publishedAt":"2021-04-19T04:07:00Z","content":"OPPOOPPO Band Style4234480\r\nSpO2\r\n24\r\n12OPPO Band Style5\r\nAndroid\r\n1210.3g2\r\nCopyright © ITmedia, Inc. All Rights Reserved."},{"source":{"id":null,"name":"Nikkei.com"},"author":"日本経済新聞社","title":"株、上値抑える現物への売り 日経先物、下落するのは東京タイム - 日本経済新聞","description":"日経平均株価の上値が重い。連日で最高値を更新する米ダウ工業株30種平均とは対照的だ。大阪取引所でほぼ一日中取引される日経平均先物の動きを東京と海外の取引時間帯で分けてみると、東京タイムでの下げが目立つ。東証の取引時間と重なり、現物株に売りが出ている証左といえる。国内外の年金基金からの売りなどが上値を抑える要因とみられている。日経平均の19日午前の終値は前週末比64円高の2万9748円だった。プ","url":"https://www.nikkei.com/article/DGXZASFL19HCR_Z10C21A4000000/","urlToImage":"https://article-image-ix.nikkei.com/https%3A%2F%2Fimgix-proxy.n8s.jp%2FDSXZQO0519396019042021000000-1.jpg?auto=format%2Ccompress&ch=Width%2CDPR&q=45&fit=crop&bg=FFFFFF&w=1200&h=630&fp_x=0.5&fp_y=0.5&fp_z=1&crop=focalpoint&ixlib=js-1.4.1&s=3f5d5f5919fc88dced69cd34c7ffb445","publishedAt":"2021-04-19T03:52:00Z","content":null},{"source":{"id":null,"name":"Impress.co.jp"},"author":null,"title":"ドコモショップでahamoの有償サポート、22日から3300円で開始予定 - ケータイ Watch","description":"NTTドコモは、4月22日からオンライン専用料金プラン「ahamo」のドコモショップでの有償サポートを提供する予定を明らかにした。料金は1回3300円（税込）の見込み。","url":"https://k-tai.watch.impress.co.jp/docs/news/1319483.html","urlToImage":"https://k-tai.watch.impress.co.jp/img/ktw/list/1319/483/sub.png","publishedAt":"2021-04-19T03:51:20Z","content":"Copyright ©2018Impress Corporation. All rights reserved."},{"source":{"id":null,"name":"Kuruma-news.jp"},"author":null,"title":"ホンダが新型電動SUVを世界初公開！ 上海で「eプロトタイプ」を発表 発売は2022年春 - くるまのニュース","description":"ホンダは2021年4月19日に、中国の上海で開催中の「2021年 上海モーターショー」において、中国初となるホンダブランドの電気自動車プロトタイプ「Honda SUV e:prototype（ホンダ・エスユーブイ・イー・プロトタイプ）」を世界初公開しました。","url":"https://kuruma-news.jp/post/368369","urlToImage":"https://kuruma-news.jp/wp-content/uploads/2022/04/20210419_honda_eprototype_004.jpg?v=1618803543","publishedAt":"2021-04-19T03:47:39Z","content":"20214192021 Honda SUV e:prototype\r\nSUV\r\n20214192021 SUVHonda SUV e:prototypee\r\n2022SUVe\r\ne2022510EV\r\n2050e\r\ne\r\nOTA3 Honda CONNECTFUN2021\r\nHonda CONNECT\r\nAIOTA\r\nCR-VPHEV\r\nPHEV2SPORT HYBRID i-MMD\r\nEVEV… [+18 chars]"},{"source":{"id":null,"name":"Gizmodo.jp"},"author":"塚本直樹","title":"ついに出る？ アップル純正MagSafe対応モバイルバッテリー - GIZMODO JAPAN","description":"Apple（アップル）純正のMagSafe対応モバイルバッテリーかもしれない画像が、ツイッター上に投稿されています。","url":"https://www.gizmodo.jp/2021/04/magsafe-battery-designed-by-california.html","urlToImage":"https://assets.media-platform.com/gizmodo/dist/images/2021/04/19/Ey_Wu_7VoAQr046-w960.jpeg","publishedAt":"2021-04-19T03:30:00Z","content":"Copyright © mediagene Inc. All Rights Reserved."},{"source":{"id":null,"name":"Response.jp"},"author":"纐纈敏也＠DAYS","title":"日産 エクストレイル 新型発表、2021年下期に市場投入…上海モーターショー2021 - レスポンス","description":"日産自動車は4月19日、上海モーターショー2021の出展概要を発表。新型『エクストレイル』や新型クロスオーバーEV『アリア』など、最先端の技術や高度なコネクティビティを搭載した多彩なラインアップを出展する。","url":"https://response.jp/article/2021/04/19/345106.html","urlToImage":"https://response.jp/imgs/ogp_f/1626177.jpg","publishedAt":"2021-04-19T03:27:45Z","content":null},{"source":{"id":null,"name":"Yahoo.co.jp"},"author":"時事通信","title":"支払日にATMで現金化 給与デジタル払いで制度案 厚労省（時事通信） - Yahoo!ニュース - Yahoo!ニュース","description":"スマートフォンの決済アプリに給与が直接入金される「デジタル払い」の解禁に向け、厚生労働省は19日、労働政策審議会（厚労相の諮問機関）の分科会に制度設計案の骨子を示した。骨子によると、デジタル払いで","url":"https://news.yahoo.co.jp/articles/0e3426cf1e5717418b795c266067b85cb381a84c","urlToImage":"https://amd-pctr.c.yimg.jp/r/iwiz-amd/20210419-00000046-jij-000-6-view.jpg","publishedAt":"2021-04-19T03:23:34Z","content":null},{"source":{"id":null,"name":"Yahoo.co.jp"},"author":"STVニュース北海道","title":"最初は「名前の頭文字」で制限 コストコ石狩２２日オープン（STVニュース北海道） - Yahoo!ニュース - Yahoo!ニュース","description":"会員制の大型倉庫型店舗を展開する「コストコ」の北海道内２号店が、石狩市にオープンします。\n\n２２日のオープンを前に一足早く売り場が公開されました。\n\n（村雨アナウンサー）「倉庫型の広い店舗には、コス","url":"https://news.yahoo.co.jp/articles/4a08fb7c1fc2b8fe77ae8bdb3a09224e4a43d060","urlToImage":"https://amd-pctr.c.yimg.jp/r/iwiz-amd/20210419-00218685-stv-000-1-thumb.jpg","publishedAt":"2021-04-19T03:23:08Z","content":null},{"source":{"id":null,"name":"Impress.co.jp"},"author":null,"title":"レクサス、新型「ES」を世界初公開 日本での発売は2021年秋ごろを予定 - Car Watch","description":"レクサス（トヨタ自動車）は4月19日、新型「ES」を「上海モーターショー2021」で世界初公開した。日本での発売は2021年秋ごろを予定している。","url":"https://car.watch.impress.co.jp/docs/rank/1319422.html","urlToImage":"https://car.watch.impress.co.jp/img/car/list/1319/422/000.jpg","publishedAt":"2021-04-19T03:17:41Z","content":"Copyright ©2018Impress Corporation. All rights reserved."},{"source":{"id":null,"name":"Response.jp"},"author":"森脇稔","title":"トヨタとスバルの共同開発EV『bZ4X』発表、2022年発売へ…上海モーターショー2021 - レスポンス","description":"◆ダイハツとスズキもbZ4Xに続くEVの共同開発プロジェクトに参画◆EV専用プラットフォーム◆スバルと共同開発した新AWDシステム","url":"https://response.jp/article/2021/04/19/345105.html","urlToImage":"https://response.jp/imgs/ogp_f/1626195.jpg","publishedAt":"2021-04-19T03:16:16Z","content":"Toyota4192021EVbZ4X\r\n2025EV15EVbZbZ7\r\nEVbZEVEVbZbeyond ZeroZero Emission\r\nbZ4XEV\r\nbZ4XEVEV\r\nbZEV1SUBARUBYD\r\nCO2 bZ4X\r\nEV\r\nbZ4XbZEVSUVEV86BRZbZ4Xe-TNGAEVAWDEV\r\nbZ4XEVD\r\n bZ4X\r\nAWD\r\nAWD\r\nbZ4XEV\r\nbZ4X20… [+1 chars]"},{"source":{"id":null,"name":"Impress.co.jp"},"author":null,"title":"mineo、20日から「AQUOS sense5G」「Redmi 9T」「moto g30」など5端末を発売 - ケータイ Watch","description":"オプテージは、MVNO型通信サービス「mineo」で、Xiaomi（シャオミ）製の「Redmi 9T」、シャープ製の「AQUOS sense5G」、モトローラ製の「moto g30」と「moto e7」のスマートフォン4端末と、レノボ製タブレット「Lenovo Tab M8」を4月20日に発売開始する。","url":"https://k-tai.watch.impress.co.jp/docs/news/1319485.html","urlToImage":"https://k-tai.watch.impress.co.jp/img/ktw/list/1319/485/mineo.jpg","publishedAt":"2021-04-19T03:14:06Z","content":"MVNOmineoXiaomiRedmi 9TAQUOS sense5Gmoto g30moto e74Lenovo Tab M8420 \r\n Redmi 9T16368682×24AQUOS sense5G464641936×24moto g3022968957×24moto e716104671×24Lenovo Tab M823760990×24 \r\n 4mineoNTTauLenovo … [+250 chars]"},{"source":{"id":null,"name":"Yahoo.co.jp"},"author":"スポーツ報知","title":"梅宮アンナ、眉アートメイクした一人娘・百々果さんと２ショット「美人親子」「大人の女性に」（スポーツ報知） - Yahoo!ニュース - Yahoo!ニュース","description":"タレントの梅宮アンナ（４８）が１９日までに自身のインスタグラムを更新。一人娘の百々果さん（１９）との親子ショットを公開した。\n\n　アンナは「東京に戻った私は、久々にももかとネイルへ」とつづり、おそ","url":"https://news.yahoo.co.jp/articles/7880a7b80311b43c62a084903bb9ff4ddbe71b32","urlToImage":"https://amd-pctr.c.yimg.jp/r/iwiz-amd/20210419-04191008-sph-000-3-view.jpg","publishedAt":"2021-04-19T03:10:05Z","content":"- - - - - - - - Copyright © 2021 The Hochi Shimbun. \r\n©Yahoo Japan"},{"source":{"id":null,"name":"Impress.co.jp"},"author":null,"title":"セブンアプリにPayPay登録で半額クーポン毎週5枚 - Impress Watch","description":"セブン-イレブン・ジャパンとPayPayは、「セブン-イレブンアプリ」にPayPayを登録すると、5週連続で対象商品の半額クーポンを5枚配布するキャンペーンを、4月28日から5月27日まで実施する。クーポンは毎週水曜日と木曜日に配布され、入手するためには前週の火曜日の時点で「セブン-イレブンアプリ」にPayPayを登録している必要がある。各クーポンは1人1枚限り。","url":"https://www.watch.impress.co.jp/docs/news/1319461.html","urlToImage":"https://www.watch.impress.co.jp/img/ipw/list/1319/461/paypay01_3x2.jpg","publishedAt":"2021-04-19T02:57:10Z","content":"-PayPay-PayPay55428527-PayPay11 \r\n --1"},{"source":{"id":null,"name":"Kuruma-news.jp"},"author":null,"title":"SUVのトヨタ「クラウン」世界初公開！ 新型「クラウンクルーガー」2021年中頃に中国で発売！ - くるまのニュース","description":"トヨタは上海モーターショー2021で、新型「クラウンクルーガー」と新型「ハイランダー」を世界初公開しました。この2台の3列シートSUVは2021年中頃に中国で発売される予定です。","url":"https://kuruma-news.jp/post/368387","urlToImage":"https://kuruma-news.jp/wp-content/uploads/2022/04/20210419_toyota_crownkluger_003.jpg?v=1618799783","publishedAt":"2021-04-19T02:49:49Z","content":"Copyright © mediavague Co., ltd.All Rights Reserved."}]}';

        if(!$jsonData) throw new Exception("Failed to get API.");
        return $jsonData;

    }catch(Exception $e){
        throw new Exception($e->getMessage());
    }
}




function GetWeatherContents($place){

    try{
        //Open Weather MapがrapidAPIで使用できなくなったので代替APIにて実装
        $ch = curl_init();
        $query = [];
        $query['lat'] = $place['lat'];
        $query['lon'] = $place['lon'];
        $query['appid'] = API_WEATHER_RAPID_KEY;

        $q = "";
        $q = '?'.http_build_query($query);

         curl_setopt_array($ch, [
            CURLOPT_URL => "https://api.openweathermap.org/data/2.5/weather".$q,
             CURLOPT_RETURNTRANSFER => true,
             CURLOPT_FOLLOWLOCATION => true,
             CURLOPT_ENCODING => "",
             CURLOPT_MAXREDIRS => 10,
             CURLOPT_TIMEOUT => 10,
             CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
             CURLOPT_CUSTOMREQUEST => "GET",
//                     CURLOPT_HTTPHEADER => [
//                         "x-rapidapi-host: community-open-weather-map.p.rapidapi.com",
//                         "x-rapidapi-key: ".API_WEATHER_RAPID_KEY
//                     ],
         ]);


        $jsonData = curl_exec($ch);
        $err = curl_error($ch);

        curl_close($ch);
        //$jsonData = '{"coord":{"lon":139.6917,"lat":35.6895},"weather":[{"id":801,"main":"Clouds","description":"few clouds","icon":"02n"}],"base":"stations","main":{"temp":291.85,"feels_like":291.55,"temp_min":290.37,"temp_max":293.15,"pressure":1014,"humidity":68},"visibility":10000,"wind":{"speed":3.09,"deg":200},"clouds":{"all":20},"dt":1618916378,"sys":{"type":1,"id":8074,"country":"JP","sunrise":1618862538,"sunset":1618910262},"timezone":32400,"id":1850144,"name":"Tokyo","cod":200}';


        if($err) throw new Exception("cURL Error # GetWeatherContents:" . $err);

        return $jsonData;

    }catch(Exception $e){
        throw new Exception($e->getMessage());
    }
}


function WriteWeatherDataForBuffer($tid, $content, $min){
    try{
        $filename =  __DIR__.'/../contents/weather_'.$tid.'.txt';
        chmod($filename, 0777);
        chown($filename, 'apache');
        $fp = fopen($filename, 'w+');
        $contents = fread($fp, filesize($filename));
        $t = new Datetime();
        $tm = $t->modify("+ ".$min." minute")->getTimestamp();

        if ($fp){
            if (!flock($fp, LOCK_EX)) throw new Exception("Failed to lock weather forecast buffer file.");

            if (fwrite($fp,  $tm.'_'.$content) === FALSE) throw new Exception("Failed to write weather forecast buffer file.");

            flock($fp, LOCK_UN);
        }

        $flag = fclose($fp);
        if(!$flag) throw new Exception("Failed to close.");
    }catch(Exception $e){
        //         var_dump($e->getMessage());exit;
        throw new Exception($e->getMessage());
    }
}



function GetThreeDayWeatherContents($place){

    try{
        //Open Weather MapがrapidAPIで使用できなくなったので代替APIにて実装
        $ch = curl_init();

        $query = [];
        $query['lat'] = $place['lat'];
        $query['lon'] = $place['lon'];
        $query['appid'] = API_WEATHER_RAPID_KEY;

        $q = "";
        $q = '?'.http_build_query($query);

        curl_setopt_array($ch, [
            CURLOPT_URL => "api.openweathermap.org/data/2.5/forecast".$q,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
//             CURLOPT_HTTPHEADER => [
//         "x-rapidapi-host: community-open-weather-map.p.rapidapi.com",
//                 "x-rapidapi-key: ".API_WEATHER_RAPID_KEY
//             ],
        ]);


        $jsonData = curl_exec($ch);
        $err = curl_error($ch);

        curl_close($ch);
        //$jsonData = '{"cod":"200","message":0,"cnt":40,"list":[{"dt":1619784000,"main":{"temp":294.45,"feels_like":293.78,"temp_min":292.48,"temp_max":294.45,"pressure":1000,"sea_level":1000,"grnd_level":998,"humidity":44,"temp_kf":1.97},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01n"}],"clouds":{"all":2},"wind":{"speed":6.86,"deg":223,"gust":9.47},"visibility":10000,"pop":0.05,"sys":{"pod":"n"},"dt_txt":"2021-04-30 12:00:00"},{"dt":1619794800,"main":{"temp":292.53,"feels_like":291.72,"temp_min":291.08,"temp_max":292.53,"pressure":1003,"sea_level":1003,"grnd_level":1000,"humidity":46,"temp_kf":1.45},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01n"}],"clouds":{"all":1},"wind":{"speed":4.43,"deg":32,"gust":6.16},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-04-30 15:00:00"},{"dt":1619805600,"main":{"temp":289.46,"feels_like":288.69,"temp_min":289.46,"temp_max":289.46,"pressure":1006,"sea_level":1006,"grnd_level":1001,"humidity":59,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01n"}],"clouds":{"all":1},"wind":{"speed":3.54,"deg":72,"gust":4.97},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-04-30 18:00:00"},{"dt":1619816400,"main":{"temp":288.98,"feels_like":288.26,"temp_min":288.98,"temp_max":288.98,"pressure":1008,"sea_level":1008,"grnd_level":1003,"humidity":63,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":1},"wind":{"speed":3.57,"deg":41,"gust":4.78},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-04-30 21:00:00"},{"dt":1619827200,"main":{"temp":291.6,"feels_like":290.83,"temp_min":291.6,"temp_max":291.6,"pressure":1008,"sea_level":1008,"grnd_level":1003,"humidity":51,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":5},"wind":{"speed":2.21,"deg":60,"gust":2.78},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-01 00:00:00"},{"dt":1619838000,"main":{"temp":294.25,"feels_like":293.43,"temp_min":294.25,"temp_max":294.25,"pressure":1006,"sea_level":1006,"grnd_level":1001,"humidity":39,"temp_kf":0},"weather":[{"id":804,"main":"Clouds","description":"overcast clouds","icon":"04d"}],"clouds":{"all":85},"wind":{"speed":5.73,"deg":179,"gust":6.11},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-01 03:00:00"},{"dt":1619848800,"main":{"temp":294.87,"feels_like":294.27,"temp_min":294.87,"temp_max":294.87,"pressure":1003,"sea_level":1003,"grnd_level":998,"humidity":45,"temp_kf":0},"weather":[{"id":804,"main":"Clouds","description":"overcast clouds","icon":"04d"}],"clouds":{"all":89},"wind":{"speed":10.48,"deg":190,"gust":16.15},"visibility":10000,"pop":0.01,"sys":{"pod":"d"},"dt_txt":"2021-05-01 06:00:00"},{"dt":1619859600,"main":{"temp":292.85,"feels_like":292.55,"temp_min":292.85,"temp_max":292.85,"pressure":1000,"sea_level":1000,"grnd_level":996,"humidity":64,"temp_kf":0},"weather":[{"id":500,"main":"Rain","description":"light rain","icon":"10d"}],"clouds":{"all":75},"wind":{"speed":10.43,"deg":196,"gust":20.13},"visibility":10000,"pop":0.88,"rain":{"3h":1.08},"sys":{"pod":"d"},"dt_txt":"2021-05-01 09:00:00"},{"dt":1619870400,"main":{"temp":292.39,"feels_like":292.28,"temp_min":292.39,"temp_max":292.39,"pressure":999,"sea_level":999,"grnd_level":994,"humidity":73,"temp_kf":0},"weather":[{"id":500,"main":"Rain","description":"light rain","icon":"10n"}],"clouds":{"all":74},"wind":{"speed":12.73,"deg":211,"gust":20.66},"visibility":10000,"pop":1,"rain":{"3h":1.92},"sys":{"pod":"n"},"dt_txt":"2021-05-01 12:00:00"},{"dt":1619881200,"main":{"temp":291.99,"feels_like":291.84,"temp_min":291.99,"temp_max":291.99,"pressure":997,"sea_level":997,"grnd_level":992,"humidity":73,"temp_kf":0},"weather":[{"id":500,"main":"Rain","description":"light rain","icon":"10n"}],"clouds":{"all":55},"wind":{"speed":10.87,"deg":209,"gust":19.81},"visibility":10000,"pop":0.86,"rain":{"3h":1.14},"sys":{"pod":"n"},"dt_txt":"2021-05-01 15:00:00"},{"dt":1619892000,"main":{"temp":289.95,"feels_like":289.3,"temp_min":289.95,"temp_max":289.95,"pressure":996,"sea_level":996,"grnd_level":991,"humidity":62,"temp_kf":0},"weather":[{"id":802,"main":"Clouds","description":"scattered clouds","icon":"03n"}],"clouds":{"all":27},"wind":{"speed":6.98,"deg":228,"gust":11.79},"visibility":10000,"pop":0.86,"sys":{"pod":"n"},"dt_txt":"2021-05-01 18:00:00"},{"dt":1619902800,"main":{"temp":289.15,"feels_like":288.19,"temp_min":289.15,"temp_max":289.15,"pressure":997,"sea_level":997,"grnd_level":992,"humidity":53,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":8},"wind":{"speed":6,"deg":215,"gust":7.41},"visibility":10000,"pop":0.03,"sys":{"pod":"d"},"dt_txt":"2021-05-01 21:00:00"},{"dt":1619913600,"main":{"temp":291.73,"feels_like":290.61,"temp_min":291.73,"temp_max":291.73,"pressure":998,"sea_level":998,"grnd_level":993,"humidity":37,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":5},"wind":{"speed":7.03,"deg":225,"gust":7.74},"visibility":10000,"pop":0.02,"sys":{"pod":"d"},"dt_txt":"2021-05-02 00:00:00"},{"dt":1619924400,"main":{"temp":293.95,"feels_like":292.92,"temp_min":293.95,"temp_max":293.95,"pressure":996,"sea_level":996,"grnd_level":992,"humidity":32,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":6},"wind":{"speed":8.3,"deg":206,"gust":8.62},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-02 03:00:00"},{"dt":1619935200,"main":{"temp":295.25,"feels_like":294.17,"temp_min":295.25,"temp_max":295.25,"pressure":995,"sea_level":995,"grnd_level":990,"humidity":25,"temp_kf":0},"weather":[{"id":801,"main":"Clouds","description":"few clouds","icon":"02d"}],"clouds":{"all":12},"wind":{"speed":8.57,"deg":247,"gust":10.41},"visibility":10000,"pop":0.02,"sys":{"pod":"d"},"dt_txt":"2021-05-02 06:00:00"},{"dt":1619946000,"main":{"temp":293.21,"feels_like":292.05,"temp_min":293.21,"temp_max":293.21,"pressure":997,"sea_level":997,"grnd_level":992,"humidity":30,"temp_kf":0},"weather":[{"id":803,"main":"Clouds","description":"broken clouds","icon":"04d"}],"clouds":{"all":51},"wind":{"speed":6.91,"deg":269,"gust":8.67},"visibility":10000,"pop":0.02,"sys":{"pod":"d"},"dt_txt":"2021-05-02 09:00:00"},{"dt":1619956800,"main":{"temp":287.44,"feels_like":286.31,"temp_min":287.44,"temp_max":287.44,"pressure":1005,"sea_level":1005,"grnd_level":1000,"humidity":53,"temp_kf":0},"weather":[{"id":500,"main":"Rain","description":"light rain","icon":"10n"}],"clouds":{"all":65},"wind":{"speed":12.33,"deg":333,"gust":15.08},"visibility":10000,"pop":0.26,"rain":{"3h":0.2},"sys":{"pod":"n"},"dt_txt":"2021-05-02 12:00:00"},{"dt":1619967600,"main":{"temp":286.46,"feels_like":284.84,"temp_min":286.46,"temp_max":286.46,"pressure":1007,"sea_level":1007,"grnd_level":1002,"humidity":38,"temp_kf":0},"weather":[{"id":500,"main":"Rain","description":"light rain","icon":"10n"}],"clouds":{"all":100},"wind":{"speed":10.01,"deg":327,"gust":12.98},"visibility":10000,"pop":0.37,"rain":{"3h":0.35},"sys":{"pod":"n"},"dt_txt":"2021-05-02 15:00:00"},{"dt":1619978400,"main":{"temp":286.23,"feels_like":284.53,"temp_min":286.23,"temp_max":286.23,"pressure":1009,"sea_level":1009,"grnd_level":1004,"humidity":36,"temp_kf":0},"weather":[{"id":803,"main":"Clouds","description":"broken clouds","icon":"04n"}],"clouds":{"all":78},"wind":{"speed":7.66,"deg":325,"gust":12.53},"visibility":10000,"pop":0.24,"sys":{"pod":"n"},"dt_txt":"2021-05-02 18:00:00"},{"dt":1619989200,"main":{"temp":286.17,"feels_like":284.44,"temp_min":286.17,"temp_max":286.17,"pressure":1012,"sea_level":1012,"grnd_level":1007,"humidity":35,"temp_kf":0},"weather":[{"id":803,"main":"Clouds","description":"broken clouds","icon":"04d"}],"clouds":{"all":74},"wind":{"speed":4.76,"deg":316,"gust":7.93},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-02 21:00:00"},{"dt":1620000000,"main":{"temp":289.79,"feels_like":288.42,"temp_min":289.79,"temp_max":289.79,"pressure":1013,"sea_level":1013,"grnd_level":1008,"humidity":35,"temp_kf":0},"weather":[{"id":803,"main":"Clouds","description":"broken clouds","icon":"04d"}],"clouds":{"all":63},"wind":{"speed":2.75,"deg":226,"gust":5.04},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-03 00:00:00"},{"dt":1620010800,"main":{"temp":292.36,"feels_like":291.17,"temp_min":292.36,"temp_max":292.36,"pressure":1012,"sea_level":1012,"grnd_level":1007,"humidity":32,"temp_kf":0},"weather":[{"id":500,"main":"Rain","description":"light rain","icon":"10d"}],"clouds":{"all":43},"wind":{"speed":5.65,"deg":192,"gust":5.88},"visibility":10000,"pop":0.28,"rain":{"3h":0.13},"sys":{"pod":"d"},"dt_txt":"2021-05-03 03:00:00"},{"dt":1620021600,"main":{"temp":292.83,"feels_like":291.66,"temp_min":292.83,"temp_max":292.83,"pressure":1011,"sea_level":1011,"grnd_level":1006,"humidity":31,"temp_kf":0},"weather":[{"id":803,"main":"Clouds","description":"broken clouds","icon":"04d"}],"clouds":{"all":63},"wind":{"speed":3.09,"deg":170,"gust":5.74},"visibility":10000,"pop":0.11,"sys":{"pod":"d"},"dt_txt":"2021-05-03 06:00:00"},{"dt":1620032400,"main":{"temp":292.65,"feels_like":291.39,"temp_min":292.65,"temp_max":292.65,"pressure":1012,"sea_level":1012,"grnd_level":1007,"humidity":28,"temp_kf":0},"weather":[{"id":804,"main":"Clouds","description":"overcast clouds","icon":"04d"}],"clouds":{"all":87},"wind":{"speed":2.89,"deg":338,"gust":6.72},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-03 09:00:00"},{"dt":1620043200,"main":{"temp":291.39,"feels_like":290.08,"temp_min":291.39,"temp_max":291.39,"pressure":1014,"sea_level":1014,"grnd_level":1009,"humidity":31,"temp_kf":0},"weather":[{"id":802,"main":"Clouds","description":"scattered clouds","icon":"03n"}],"clouds":{"all":50},"wind":{"speed":2.15,"deg":302,"gust":5.54},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-05-03 12:00:00"},{"dt":1620054000,"main":{"temp":290.06,"feels_like":288.9,"temp_min":290.06,"temp_max":290.06,"pressure":1015,"sea_level":1015,"grnd_level":1010,"humidity":42,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01n"}],"clouds":{"all":0},"wind":{"speed":3.09,"deg":237,"gust":4.78},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-05-03 15:00:00"},{"dt":1620064800,"main":{"temp":288.9,"feels_like":287.58,"temp_min":288.9,"temp_max":288.9,"pressure":1016,"sea_level":1016,"grnd_level":1011,"humidity":40,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01n"}],"clouds":{"all":0},"wind":{"speed":3.13,"deg":246,"gust":6.58},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-05-03 18:00:00"},{"dt":1620075600,"main":{"temp":288.94,"feels_like":287.67,"temp_min":288.94,"temp_max":288.94,"pressure":1017,"sea_level":1017,"grnd_level":1012,"humidity":42,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":3},"wind":{"speed":2.53,"deg":227,"gust":4.65},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-03 21:00:00"},{"dt":1620086400,"main":{"temp":292.41,"feels_like":291.33,"temp_min":292.41,"temp_max":292.41,"pressure":1017,"sea_level":1017,"grnd_level":1013,"humidity":36,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":2},"wind":{"speed":2.67,"deg":186,"gust":3.61},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-04 00:00:00"},{"dt":1620097200,"main":{"temp":295.49,"feels_like":294.46,"temp_min":295.49,"temp_max":295.49,"pressure":1016,"sea_level":1016,"grnd_level":1011,"humidity":26,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":0},"wind":{"speed":3.66,"deg":170,"gust":4.56},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-04 03:00:00"},{"dt":1620108000,"main":{"temp":296.37,"feels_like":295.4,"temp_min":296.37,"temp_max":296.37,"pressure":1015,"sea_level":1015,"grnd_level":1010,"humidity":25,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":0},"wind":{"speed":5,"deg":166,"gust":4.28},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-04 06:00:00"},{"dt":1620118800,"main":{"temp":294.83,"feels_like":293.97,"temp_min":294.83,"temp_max":294.83,"pressure":1015,"sea_level":1015,"grnd_level":1010,"humidity":35,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01d"}],"clouds":{"all":0},"wind":{"speed":5.27,"deg":179,"gust":5.67},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-04 09:00:00"},{"dt":1620129600,"main":{"temp":293.32,"feels_like":292.54,"temp_min":293.32,"temp_max":293.32,"pressure":1016,"sea_level":1016,"grnd_level":1011,"humidity":44,"temp_kf":0},"weather":[{"id":800,"main":"Clear","description":"clear sky","icon":"01n"}],"clouds":{"all":0},"wind":{"speed":4.85,"deg":198,"gust":6.11},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-05-04 12:00:00"},{"dt":1620140400,"main":{"temp":291.71,"feels_like":291.06,"temp_min":291.71,"temp_max":291.71,"pressure":1016,"sea_level":1016,"grnd_level":1011,"humidity":55,"temp_kf":0},"weather":[{"id":802,"main":"Clouds","description":"scattered clouds","icon":"03n"}],"clouds":{"all":26},"wind":{"speed":4.53,"deg":213,"gust":5.9},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-05-04 15:00:00"},{"dt":1620151200,"main":{"temp":290.91,"feels_like":290.26,"temp_min":290.91,"temp_max":290.91,"pressure":1016,"sea_level":1016,"grnd_level":1011,"humidity":58,"temp_kf":0},"weather":[{"id":801,"main":"Clouds","description":"few clouds","icon":"02n"}],"clouds":{"all":23},"wind":{"speed":4.43,"deg":218,"gust":6.15},"visibility":10000,"pop":0,"sys":{"pod":"n"},"dt_txt":"2021-05-04 18:00:00"},{"dt":1620162000,"main":{"temp":290.58,"feels_like":289.97,"temp_min":290.58,"temp_max":290.58,"pressure":1016,"sea_level":1016,"grnd_level":1011,"humidity":61,"temp_kf":0},"weather":[{"id":803,"main":"Clouds","description":"broken clouds","icon":"04d"}],"clouds":{"all":66},"wind":{"speed":5.68,"deg":214,"gust":8.06},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-04 21:00:00"},{"dt":1620172800,"main":{"temp":293.97,"feels_like":293.26,"temp_min":293.97,"temp_max":293.97,"pressure":1015,"sea_level":1015,"grnd_level":1010,"humidity":44,"temp_kf":0},"weather":[{"id":803,"main":"Clouds","description":"broken clouds","icon":"04d"}],"clouds":{"all":83},"wind":{"speed":9.48,"deg":209,"gust":12},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-05 00:00:00"},{"dt":1620183600,"main":{"temp":296.07,"feels_like":295.41,"temp_min":296.07,"temp_max":296.07,"pressure":1013,"sea_level":1013,"grnd_level":1008,"humidity":38,"temp_kf":0},"weather":[{"id":804,"main":"Clouds","description":"overcast clouds","icon":"04d"}],"clouds":{"all":99},"wind":{"speed":12.66,"deg":206,"gust":14.89},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-05 03:00:00"},{"dt":1620194400,"main":{"temp":294.92,"feels_like":294.43,"temp_min":294.92,"temp_max":294.92,"pressure":1012,"sea_level":1012,"grnd_level":1007,"humidity":49,"temp_kf":0},"weather":[{"id":804,"main":"Clouds","description":"overcast clouds","icon":"04d"}],"clouds":{"all":100},"wind":{"speed":12.29,"deg":202,"gust":14.91},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-05 06:00:00"},{"dt":1620205200,"main":{"temp":293.22,"feels_like":292.69,"temp_min":293.22,"temp_max":293.22,"pressure":1013,"sea_level":1013,"grnd_level":1008,"humidity":54,"temp_kf":0},"weather":[{"id":804,"main":"Clouds","description":"overcast clouds","icon":"04d"}],"clouds":{"all":100},"wind":{"speed":11.45,"deg":206,"gust":14.93},"visibility":10000,"pop":0,"sys":{"pod":"d"},"dt_txt":"2021-05-05 09:00:00"}],"city":{"id":1850144,"name":"Tokyo","coord":{"lat":35.6895,"lon":139.6917},"country":"JP","population":12445327,"timezone":32400,"sunrise":1619725834,"sunset":1619774771}}';


        if($err) throw new Exception("cURL Error # GetThreeDayWeatherContents:" . $err);

        return $jsonData;

    }catch(Exception $e){
        throw new Exception($e->getMessage());
    }
}


function WriteThreeWeatherDataForBuffer($tid, $content, $min){
    try{
        $filename =  __DIR__.'/../contents/weather3hour_'.$tid.'.txt';
        chmod($filename, 0777);
        chown($filename, 'apache');
        $fp = fopen($filename, 'w+');
        $contents = fread($fp, filesize($filename));
        $t = new Datetime();
        $tm = $t->modify("+ ".$min." minute")->getTimestamp();

        if ($fp){
            if (!flock($fp, LOCK_EX)) throw new Exception("Failed to lock weather forecast buffer file.");

            if (fwrite($fp,  $tm.'_'.$content) === FALSE) throw new Exception("Failed to write weather forecast buffer file.");

            flock($fp, LOCK_UN);
        }

        $flag = fclose($fp);
        if(!$flag) throw new Exception("Failed to close.");

    }catch(Exception $e){
        //         var_dump($e->getMessage());exit;
        throw new Exception($e->getMessage());
    }
}



// function AddDeviceData($tid, $apiData, $MDL){
//     //システムに登録されているデバイス一覧を取得
//     $alrdList = [];
//     $d_row = $MDL->GetDeviceTblByTenantID($tid);
//     if($d_row){
//         foreach($d_row as $dK => $dV){
//             $alrdList[] = $dV['device_id'];
//         }
//     }

//     if($apiData){
//         foreach($apiData as $apK => $apV){
//             if(!in_array($apV['DeviceId'], $alrdList, TRUE)){
//                 //システムに登録されていないデバイスであれば登録
//                 $prm = [
//                     'device_id' => $apV['DeviceId'],
//                     'tenant_id' => $tid,
//                     'name' => $apV['Name'],
//                     'type' => 'camera',
//                     'status' => '1',
//                 ];
//                 $ins_ret = $MDL->InsertDeviceData($prm);
//             }
//         }
//     }

// }



function WriteDeviceDataForBuffer($tid, $content, $min){
    try{
        //TODO 照明及び空調デバイスの情報も取得・格納

        $t = new Datetime();
        $timestr = $t->format("Ym");
        $filename =  __DIR__.'/../contents/devicedata_'.$tid.'_'.$timestr.'.txt';
        chmod($filename, 0777);
        chown($filename, 'apache');
        $fp = fopen($filename, 'a+');

        $write = implode("\n", array_map('json_encode', $content));
        $write .= "\n";

        if (!is_null($write) && $fp){
            if (!flock($fp, LOCK_EX)) throw new Exception("Failed to lock device information buffer file.");
            if (fwrite($fp, $write) === FALSE) throw new Exception("Failed to write device information buffer file.");
            flock($fp, LOCK_UN);
        }

        $flag = fclose($fp);
        if(!$flag) throw new Exception("Failed to close.");
    }catch(Exception $e){
        //         var_dump($e->getMessage());exit;
        throw new Exception($e->getMessage());
    }
}

?>